/**
 * \file sdc_op_sign_verify .c
 *
 * \brief Functions for sign and verify
 * Please note : the implementation is split into different operations
 * instead of splitting it in common, convenience and advanced functions
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <string.h>
#include <sdc_op_common.h>
#include <sdc_op_adv.h>
#include <sdc_op_conv.h>
#include <sdc_random.h>
#include <private/sdc_arch.h>
#include <private/sdc_intern.h>

/* Definitions types and defaults */

static const char *sdc_sign_verify_alg_names[] = {
    [SDC_SIGNVER_ALG_HMAC] = "HMAC",
    [SDC_SIGNVER_ALG_RSA] = "RSA"
};

static const char *sdc_sign_verify_hash_names[] = {
    [SDC_SIGNVER_HASH_MD5] = "MD5",
    [SDC_SIGNVER_HASH_SHA1] = "SHA1",
    [SDC_SIGNVER_HASH_SHA224] = "SHA224",
    [SDC_SIGNVER_HASH_SHA256] = "SHA256",
    [SDC_SIGNVER_HASH_SHA384] = "SHA384",
    [SDC_SIGNVER_HASH_SHA512] = "SHA512",
};

/* Functions */

/**
 * \brief Checks common to sign and verify
 *
 * Check session, type and in data
 * Provide \ref sdc_sign_verify_desc_t of type
 */
static sdc_error_t sdc_sign_verify_common_checks_defaults(
    sdc_error_t error_init,
    sdc_session_t *session,
    const sdc_sign_verify_type_t *type,
    sdc_sign_verify_desc_t *internal_desc,
    const uint8_t *in_data, const size_t in_len)
{

    sdc_error_t err = error_init;

    if (SDC_OK != sdc_intern_check_data_input_buffer(in_data, in_len))
        err = SDC_IN_DATA_INVALID;

    if (!type)
        err = SDC_ALG_MODE_INVALID;

    if (!session)
        err = SDC_SESSION_INVALID;

    if (err == SDC_OK)
        err = sdc_sign_verify_desc_fill (session, type, internal_desc);

    return err;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_desc_get_max_chunk_len(const sdc_sign_verify_desc_t *desc,
                                                       size_t *max_val)
{
    size_t max_chunk;

    if (!desc)
        return SDC_INVALID_PARAMETER;

    if (max_val) {
        max_chunk = desc->data.max_chunk_len;
        if (desc->data.chunk_len_aligned) {
            max_chunk -= (max_chunk % desc->data.block_len);
        }

        if (max_chunk == 0)
            return SDC_INTERNAL_ERROR;

        *max_val = max_chunk;
    }

    return SDC_OK;
}

static sdc_error_t sdc_sign_verify_common_check_tag(const sdc_sign_verify_desc_t *desc, size_t tag_len)
{
    if (SDC_OK != sdc_intern_range_min_max_mod_check(tag_len, &(desc->tag)))
        return SDC_TAG_DATA_INVALID;

    return SDC_OK;
}

/**
 * \brief call min max checks for in data and iv
 */
static sdc_error_t sdc_sign_verify_common_check_iv(const sdc_sign_verify_desc_t *desc, size_t iv_len )
{
    if (sdc_intern_range_min_max_mod_check(iv_len, &(desc->iv)) != SDC_OK)
        return SDC_IV_INVALID;

    return SDC_OK;
}

/**
 * \brief call min max checks for in data sig and iv
 */
static sdc_error_t sdc_sign_verify_common_check_iv_tag_in_length(const sdc_sign_verify_desc_t *desc,
                                                                 size_t in_len, size_t iv_len, size_t tag_len)
{
    sdc_error_t err;

    err = sdc_intern_input_only_check_min_max_align(in_len,
                                                    &(desc->data));
    if(err != SDC_OK)
        return err;

    err = sdc_sign_verify_common_check_iv(desc, iv_len);
    if(err != SDC_OK)
        return err;

    return sdc_sign_verify_common_check_tag(desc, tag_len);
}

static sdc_error_t sdc_common_signverify_init(sdc_session_t *session,
                                              const sdc_sign_verify_type_t *type,
                                              const sdc_sign_verify_desc_t *desc,
                                              const uint8_t *iv, size_t iv_len,
                                              bool is_sign)
{
    session->unaligned_buffer_filllevel = 0;
    session->inlen_cnt = 0;

    if (!desc->supports_iuf)
        return SDC_OP_NOT_SUPPORTED;

    if (is_sign)
        return sdc_arch_sign_init(session, type, desc, iv, iv_len);

    return sdc_arch_verify_init(session, type, desc, iv, iv_len);
}

static sdc_error_t sdc_common_signverify_update(sdc_session_t *session,
                                                const sdc_sign_verify_type_t *type,
                                                const sdc_sign_verify_desc_t *desc,
                                                const uint8_t *in_data, const size_t in_data_len,  bool is_sign)
{
    sdc_error_t err = SDC_OK;
    size_t block_len = desc->data.block_len;
    size_t chunk_len = desc->data.max_chunk_len;
    bool chunk_len_aligned = desc->data.chunk_len_aligned;
    const uint8_t *next_in_data;
    size_t in_len_remaining;
    size_t unaligned_buf_rem;
    uint8_t *unaligned_buf_next;
    size_t min_no_proc_len;
    size_t process_in_len;
    size_t remainder;

    if (session->unaligned_buffer_filllevel > block_len)
        err = SDC_INTERNAL_ERROR;

    /* Check that processing additional data won't exceed max */
    if (err == SDC_OK)
        err = sdc_intern_input_only_check_update_wont_exceed_max(
                session->inlen_cnt,
                in_data_len,
                &(desc->data));

    next_in_data = in_data;
    in_len_remaining = in_data_len;

    /* handle previous unaligned data first */
    if ((err == SDC_OK) && (session->unaligned_buffer_filllevel > 0)) {
        /* if unaligned_buffer_filllevel > 0 we obviously need to alignment */
        unaligned_buf_rem = block_len - session->unaligned_buffer_filllevel;

        if (!chunk_len_aligned) {
            /*
             * This must not happen
             * Probably someone has corrupted the control structures
             */
            err = SDC_INTERNAL_ERROR;
        } else {
            if (in_len_remaining >= unaligned_buf_rem)
            {
                unaligned_buf_next = session->unaligned_buffer;
                unaligned_buf_next += session->unaligned_buffer_filllevel;

                /* afterwards we have one block in the buffer */
                memcpy(unaligned_buf_next,
                       next_in_data,
                       unaligned_buf_rem);

                /* process the unaligned part */
                if (is_sign) {
                    err = sdc_arch_sign_update(session, type, desc,
                                                  session->unaligned_buffer, block_len);
                } else {
                    err = sdc_arch_verify_update(session, type, desc,
                                                  session->unaligned_buffer, block_len);
                }
                if (err == SDC_OK) {
                    /* update pointers + lengths for remaining data */
                    next_in_data += unaligned_buf_rem;
                    in_len_remaining -= unaligned_buf_rem;

                    /* all data handled */
                    session->unaligned_buffer_filllevel = 0;
                }
            }
            /*
             * else case
             *       i.e. in_len_remaining < unaligned_buf_rem
             * will be handled by "append remaining data to unaligned_buffer"
             *
             * In this case
             *      in_len_remaining is < block_len as well and chunk_len_aligned true.
             *
             * Otherwise there is no way (beside fooling around with internal
             * structs) that data has been added to unaligned_buffer in the
             * previous call.
             *
             * Note: the while loop won't process any data in this case
             */
        }
    }

    /* smaller or equal length need to be handled using unaligned_buffer */
    min_no_proc_len = 0;
    if (chunk_len_aligned)
        min_no_proc_len = block_len - 1;

    while ((err == SDC_OK) && (in_len_remaining > min_no_proc_len)) {
        process_in_len = in_len_remaining;

        if (process_in_len > chunk_len)
            process_in_len = chunk_len;

        if (chunk_len_aligned) {
            /* align */
            remainder = (process_in_len % block_len);
            if (remainder != 0) {
                process_in_len -= remainder;
            }
        }

        /* process the next chunk of data */
        if (is_sign) {
            err = sdc_arch_sign_update(session, type, desc,
                                          next_in_data, process_in_len);
        } else {
            err = sdc_arch_verify_update(session, type, desc,
                                          next_in_data, process_in_len);
        }

        if (err == SDC_OK) {
            /* update pointers + lengths for remaining data */
            next_in_data += process_in_len;
            in_len_remaining -= process_in_len;
        }
    }

    /* append remaining data to unaligned_buffer */
    if ((err == SDC_OK) && (in_len_remaining > 0)) {
        unaligned_buf_rem = block_len - session->unaligned_buffer_filllevel;

        if (in_len_remaining > unaligned_buf_rem) {
            /* this must not happen */
            err = SDC_INTERNAL_ERROR;
        } else {
            unaligned_buf_next = session->unaligned_buffer;
            size_t fill_level = session->unaligned_buffer_filllevel;
            unaligned_buf_next += fill_level;

            /* append to the end of the unaligned buffer */
            memcpy(unaligned_buf_next,
                   next_in_data,
                   in_len_remaining);
            session->unaligned_buffer_filllevel += in_len_remaining;

            /* no need to update in_data_remaining */
            in_len_remaining = 0;
        }
    }

    if (err != SDC_OK) {
        /* clear confidential data in case of error */
        sdc_intern_clear_session_confidential(session);
    } else {
        /* add the current input data length */
        session->inlen_cnt += in_data_len;
    }

    return err;
}

static sdc_error_t sdc_common_signverify_finalize(sdc_session_t *session,
                                                  const sdc_sign_verify_type_t *type,
                                                  const sdc_sign_verify_desc_t *desc,
                                                  uint8_t *tag_out_data, size_t tag_out_data_len,
                                                  const uint8_t *tag_in_data, size_t tag_in_data_len,
                                                  bool is_sign)
{
    sdc_error_t err = SDC_OK;
    size_t unaligned_data_len;
    sdc_padding_t padding = desc->data.padding;
    size_t block_len = desc->data.block_len;
    bool padded = false;

    unaligned_data_len = session->unaligned_buffer_filllevel;

    /* Check that total provided length fits to min/max */
    err = sdc_intern_input_only_check_min_max_align(
            session->inlen_cnt,
            &(desc->data));

    if ((padding != SDC_PADDING_INTERNAL) &&
        (padding != SDC_PADDING_NO) &&
        (padding != SDC_PADDING_HIDDEN))
        padded = true;

    if ((err == SDC_OK) && (padded)) {
        err = sdc_intern_pad(padding,
                             session->unaligned_buffer,
                             unaligned_data_len,
                             block_len);
        unaligned_data_len = block_len;
    }

    if (is_sign) {
        if (err == SDC_OK)
            err = sdc_arch_sign_finalize(session, type, desc,
                                            session->unaligned_buffer,
                                            unaligned_data_len,
                                            tag_out_data, tag_out_data_len);
    } else {

        if (err == SDC_OK)
            err = sdc_arch_verify_finalize(session, type, desc,
                                            session->unaligned_buffer,
                                            unaligned_data_len,
                                            tag_in_data, tag_in_data_len);
    }

    sdc_intern_clear_session_confidential(session);

    return err;
}


/**
 * \brief common implementation of sign using architecture dependent
 * init, update, finalize functions
 *
 * This function will be used when the architecture dependent part does not provide
 * optimized sign functionality
 *
 * \todo add functionality
 *
 */
static sdc_error_t sdc_common_sign(sdc_session_t *session,
                                   const sdc_sign_verify_type_t *type,
                                   const sdc_sign_verify_desc_t *desc,
                                   const uint8_t *in_data,
                                   const size_t in_data_len,
                                   const uint8_t *iv, const size_t iv_len,
                                   uint8_t *tag_out_data, const size_t tag_out_len)
{
    sdc_error_t err;

    err = sdc_common_signverify_init(session, type, desc, iv, iv_len, true);
    if (err == SDC_OK) {
        err = sdc_common_signverify_update(session, type, desc, in_data, in_data_len, true);
    }

    if (err == SDC_OK) {
        err = sdc_common_signverify_finalize(session, type, desc, tag_out_data, tag_out_len, NULL, 0, true);
    }

    return err;

}

/**
 * \brief common implementation of verify using architecture dependent
 * init, update, finalize functions
 *
 * This function will be used when the architecture dependent part does not provide
 * optimized verify functionality
 *
 * \todo add functionality
 *
 */
static sdc_error_t sdc_common_verify(sdc_session_t *session,
                                     const sdc_sign_verify_type_t *type,
                                     const sdc_sign_verify_desc_t *desc,
                                     const uint8_t *in_data,
                                     const size_t in_data_len,
                                     const uint8_t *iv, const size_t iv_len,
                                     const uint8_t *tag_in_data, const size_t tag_in_len)
{

    sdc_error_t err;

    err = sdc_common_signverify_init(session, type, desc, iv, iv_len, false);
    if (err == SDC_OK) {
        err = sdc_common_signverify_update(session, type, desc, in_data, in_data_len, false);
    }

    if (err == SDC_OK) {
        err = sdc_common_signverify_finalize(session, type, desc, NULL, 0, tag_in_data, tag_in_len, false);
    }
    return err;
}

/**
 * \brief select if architecture specific or common version is used
 */
static sdc_error_t sdc_sign_selector(sdc_session_t *session,
                                     const sdc_sign_verify_type_t *type,
                                     const sdc_sign_verify_desc_t *desc,
                                     const uint8_t *in_data,
                                     const size_t in_data_len,
                                     const uint8_t *iv, const size_t iv_len,
                                     uint8_t *tag_out_data, const size_t tag_out_len)
{

    sdc_error_t err;

    err = sdc_arch_sign(session,
                        type,
                        desc,
                        in_data,
                        in_data_len,
                        iv, iv_len,
                        tag_out_data, tag_out_len);

    if (err == SDC_NOT_SUPPORTED) {
        err = sdc_common_sign(session,
                              type,
                              desc,
                              in_data,
                              in_data_len,
                              iv, iv_len,
                              tag_out_data, tag_out_len);
    }

    return err;
}

/**
 * \brief select if architecture specific or common version is used
 */
static sdc_error_t sdc_verify_selector(sdc_session_t *session,
                                       const sdc_sign_verify_type_t *type,
                                       const sdc_sign_verify_desc_t *desc,
                                       const uint8_t *in_data,
                                       const size_t in_data_len,
                                       const uint8_t *iv, const size_t iv_len,
                                       const uint8_t *tag_in_data, const size_t tag_in_len)
{

    sdc_error_t err;

    err = sdc_arch_verify(session,
                          type,
                          desc,
                          in_data,
                          in_data_len,
                          iv, iv_len,
                          tag_in_data, tag_in_len);

    if (err == SDC_NOT_SUPPORTED) {
        err = sdc_common_verify(session,
                                type,
                                desc,
                                in_data,
                                in_data_len,
                                iv, iv_len,
                                tag_in_data, tag_in_len);
    }

    return err;
}


sdc_error_t sdc_sign(sdc_session_t *session,
                     const sdc_sign_verify_type_t *type,
                     const uint8_t *in_data, const size_t in_len,
                     uint8_t **iv, size_t *iv_len,
                     uint8_t **tag_data, size_t *tag_len)
{

    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_sign_verify_desc_t internal_desc;

    bool explicit_iv;
    uint8_t *internal_iv;
    size_t internal_iv_len;
    uint8_t *internal_tag = NULL;
    size_t internal_tag_len;

    /* initialize - in case failing with error we want to return
     * buffer pointer NULL and buffer len 0 for all not explicitly specified buffers
     */
    if (SDC_OK != sdc_intern_check_init_dgst_tag_iv_output_buffer(tag_data, tag_len, &internal_tag_len, SDC_TAG_USE_DEFAULT, NULL)) {
        err = SDC_TAG_DATA_INVALID;
    }

    if (SDC_OK != sdc_intern_check_init_dgst_tag_iv_output_buffer(iv, iv_len, &internal_iv_len, SDC_IV_USE_DEFAULT, &explicit_iv)) {
        err = SDC_IV_INVALID;
    }

    err = sdc_sign_verify_common_checks_defaults(err,
                                                 session,
                                                 type, &internal_desc,
                                                 in_data, in_len);
    if (err != SDC_OK)
        return err;

    if (explicit_iv) {
        internal_iv = *iv;
    } else {
        if (internal_iv_len == SDC_IV_USE_DEFAULT) {
            internal_iv_len = internal_desc.iv.dflt;
        }
        internal_iv = NULL;
        /* there might be formats without IV */
        if (internal_iv_len != 0) {
            err = sdc_random_gen_buffer(session, internal_iv_len, &internal_iv);
        }
    }

    if (internal_tag_len == SDC_TAG_USE_DEFAULT) {
        internal_tag_len = internal_desc.tag.dflt;
    }

    if (err == SDC_OK) {
        err = sdc_sign_verify_common_check_iv_tag_in_length(&internal_desc, in_len, internal_iv_len, internal_tag_len);
    }

    if (err == SDC_OK) {
        if (internal_tag_len != 0) {
            internal_tag = malloc(internal_tag_len);
            if (!internal_tag)
                err = SDC_NO_MEM;
        }
    }

    if (err == SDC_OK) {
        err = sdc_sign_selector(session,
                                type,
                                &internal_desc,
                                in_data,
                                in_len,
                                internal_iv, internal_iv_len,
                                internal_tag, internal_tag_len);
    }

    if (err == SDC_OK) {
        if (!explicit_iv) {
            *iv = internal_iv;
            *iv_len = internal_iv_len;
        }
        *tag_data = internal_tag;
        *tag_len = internal_tag_len;
    } else {
        /* clean allocated memory */
        if (!explicit_iv)
            free (internal_iv);
        free(internal_tag);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_SIGN, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_verify(sdc_session_t *session,
                       const sdc_sign_verify_type_t *type,
                       const uint8_t *in_data, const size_t in_len,
                       const uint8_t *iv, const size_t iv_len,
                       const uint8_t *tag_data, const size_t tag_len)
{

    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_sign_verify_desc_t internal_desc;

    if (SDC_OK != sdc_intern_check_tag_iv_input_buffer(tag_data, tag_len, SDC_TAG_USE_DEFAULT))
        err = SDC_TAG_DATA_INVALID;

    if (SDC_OK != sdc_intern_check_tag_iv_input_buffer(iv, iv_len, SDC_IV_USE_DEFAULT))
        err = SDC_IV_INVALID;

    err = sdc_sign_verify_common_checks_defaults(err,
                                                 session,
                                                 type, &internal_desc,
                                                 in_data, in_len);

    if (err != SDC_OK)
        return err;

    err = sdc_sign_verify_common_check_iv_tag_in_length(&internal_desc, in_len, iv_len, tag_len);

    if (err == SDC_OK) {
        err = sdc_verify_selector(session,
                                  type,
                                  &internal_desc,
                                  in_data,
                                  in_len,
                                  iv, iv_len,
                                  tag_data, tag_len);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_VERIFY, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}



sdc_error_t sdc_sign_formatted_extended(sdc_session_t *session,
                                        const sdc_sign_verify_type_t *type,
                                        const uint8_t *in_data, const size_t in_len,
                                        const uint8_t *iv, const size_t iv_len,
                                        const size_t tag_len,
                                        uint8_t **formatted_data, size_t *formatted_len)
{

    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_sign_verify_desc_t internal_desc;
    size_t internal_iv_len;
    bool explicit_iv;
    size_t internal_tag_len;
    sdc_form_header_generic_t form_header;
    sdc_form_header_sign_verify_generic_t *form_sign;
    uint8_t *internal_formatted_buffer = NULL;
    uint8_t *iv_tmp;

    /* if the iv is externally determined the len has to be set externally too */
    explicit_iv = (iv != NULL);
    if (explicit_iv && (iv_len == SDC_IV_USE_DEFAULT))
        err = SDC_IV_INVALID;

    if (SDC_OK != sdc_intern_check_init_data_output_buffer(formatted_data, formatted_len))
        err = SDC_FORMATTED_DATA_INVALID;

    err = sdc_sign_verify_common_checks_defaults(err,
                                                 session,
                                                 type, &internal_desc,
                                                 in_data, in_len);

    if (err != SDC_OK)
        return err;

    /* get defaults if not set by application */
    internal_iv_len = iv_len;
    if (internal_iv_len == SDC_IV_USE_DEFAULT) {
        internal_iv_len = internal_desc.iv.dflt;
    }
    internal_tag_len = tag_len;
    if (internal_tag_len == SDC_TAG_USE_DEFAULT) {
        internal_tag_len = internal_desc.tag.dflt;
    }

    err = sdc_sign_verify_common_check_iv_tag_in_length(&internal_desc, in_len, internal_iv_len, internal_tag_len);

    if (err == SDC_OK) {
        uint64_t opt_bmsk;

        err = sdc_sign_verify_type_get_opt_bmsk(type, &opt_bmsk);

        /* using formatted with additional options is not allowed */
        if ((err == SDC_OK) && (opt_bmsk != 0))
            err = SDC_OP_NOT_SUPPORTED;
    }

    if (err == SDC_OK) {
        sdc_inter_form_header_init(&form_header);

        err = sdc_intern_sign_verify_formatted_fill_header (session,
                                                            type,
                                                            in_len,
                                                            internal_iv_len,
                                                            internal_tag_len,
                                                            &form_header);

        if (err == SDC_OK) {
            internal_formatted_buffer = malloc(form_header.overall_formatted_len);
            if (internal_formatted_buffer == NULL)
                err = SDC_NO_MEM;
        }

        if (err == SDC_OK) {
            err = sdc_intern_sign_verify_formatted_update_pointers(&form_header,
                                                                   internal_formatted_buffer);
        }

        if (err == SDC_OK) {
            form_sign = &form_header.sign_verify;
            err = sdc_intern_formatted_write_header(&form_header,
                                                    internal_formatted_buffer);

            if (err == SDC_OK) {
                if (internal_iv_len != 0) {
                    if (explicit_iv) {
                        /* copy explicitly specified iv */
                        memcpy (form_sign->iv, iv, iv_len);
                    } else {
                        iv_tmp = NULL;
                        err = sdc_random_gen_buffer(session, internal_iv_len, &iv_tmp);
                        if (err == SDC_OK) {
                            memcpy (form_sign->iv, iv_tmp, internal_iv_len);
                        }
                        if (iv_tmp != NULL)
                            free(iv_tmp);
                    }
                }
            }
            if ((err == SDC_OK) && (in_len != 0)) {
                /* copy data (plain) */
                memcpy (form_sign->data, in_data, in_len);
            }

            if (err == SDC_OK) {
                err = sdc_sign_selector(session,
                                        type,
                                        &internal_desc,
                                        form_sign->data, form_sign->data_len,
                                        form_sign->iv, form_sign->iv_len,
                                        form_sign->tag, form_sign->tag_len);
            }
        }

        if (err == SDC_OK) {
            *formatted_data = internal_formatted_buffer;
            *formatted_len = form_header.overall_formatted_len;
        } else {
            if (internal_formatted_buffer)
                free (internal_formatted_buffer);
        }

        sdc_inter_form_header_free(&form_header);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_SIGN, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_sign_formatted(sdc_session_t *session,
                               const sdc_sign_verify_type_t *type,
                               const uint8_t *in_data, const size_t in_len,
                               uint8_t **formatted_data, size_t *formatted_len)
{
    return sdc_sign_formatted_extended (session,
                                        type,
                                        in_data, in_len,
                                        NULL, SDC_IV_USE_DEFAULT,
                                        SDC_TAG_USE_DEFAULT,
                                        formatted_data, formatted_len);
}

static sdc_error_t sdc_intern_check_init_const_data_output_buffer (const uint8_t **data, size_t *len)
{
    sdc_error_t err = SDC_OK;

    if (!data) {
        err = SDC_INVALID_PARAMETER;
    } else {
        if (*data) {
            err = SDC_INVALID_PARAMETER;
            *data = NULL;
        }
    }

    if (!len) {
        err = SDC_INVALID_PARAMETER;
    } else {
        *len = 0;
    }

    return err;
}

sdc_error_t sdc_verify_formatted(sdc_session_t *session,
                                 const sdc_sign_verify_type_t *type,
                                 const uint8_t *formatted_data, const size_t formatted_len,
                                 const uint8_t **out_data, size_t *out_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_form_header_generic_t form_header;
    sdc_form_header_sign_verify_generic_t *form_sign = NULL;
    sdc_sign_verify_desc_t desc;
    const uint8_t *internal_data = NULL;
    size_t internal_data_len = 0;
    sdc_sign_verify_alg_t mac;
    sdc_sign_verify_hash_t hash;

    if (SDC_OK != sdc_intern_check_init_const_data_output_buffer(out_data, out_len))
        err = SDC_OUT_DATA_INVALID;

    if (SDC_OK != sdc_intern_check_data_input_buffer(formatted_data, formatted_len))
        err = SDC_FORMATTED_DATA_INVALID;

    if (!session)
        err = SDC_SESSION_INVALID;

    if (err != SDC_OK)
        return err;

    err = sdc_intern_formatted_read_header (
        &form_header,
        formatted_data, formatted_len);

    if (err == SDC_OK) {
        if (form_header.operation != SDC_FORMATTED_TYPE_SIGN_VERIFY)
            err = SDC_FORMATTED_DATA_INVALID;
    }

    if (err == SDC_OK) {
        form_sign = &form_header.sign_verify;
        /* we need to typecast as the same pointers in sign struct are used for
         * read and write. Before calling verify we will make it const again */
        err = sdc_intern_sign_verify_formatted_update_pointers(
            &form_header,
            (uint8_t*)formatted_data);

        /* check if given type matches the one from the formatted data */
        if (err == SDC_OK) {
            err = sdc_sign_verify_type_get_alg(type, &mac);

            if ((err == SDC_OK) && (mac != form_sign->alg))
                err = SDC_ALG_MODE_INVALID;
        }
        if (err == SDC_OK) {
            err = sdc_sign_verify_type_get_hash(type, &hash);

            if ((err == SDC_OK) && (hash != form_sign->hash))
                err = SDC_ALG_MODE_INVALID;
        }

        if (err == SDC_OK) {
            uint64_t opt_bmsk;

            err = sdc_sign_verify_type_get_opt_bmsk(type, &opt_bmsk);

            /* using formatted with additional options is not allowed */
            if ((err == SDC_OK) && (opt_bmsk != 0))
                err = SDC_OP_NOT_SUPPORTED;
        }

        if (err == SDC_OK) {
            err = sdc_sign_verify_desc_fill (session, type, &desc);

            if (err == SDC_OK) {
                err = sdc_sign_verify_common_check_iv_tag_in_length(&desc,
                                                                    form_sign->data_len,
                                                                    form_sign->iv_len,
                                                                    form_sign->tag_len);

                /* allocate output buffer */
                if (err == SDC_OK) {
                    internal_data_len = form_sign->data_len;
                    internal_data = form_sign->data;
                }

                if (err == SDC_OK) {
                    err = sdc_verify_selector(session,
                                              type,
                                              &desc,
                                              internal_data,
                                              internal_data_len,
                                              (const uint8_t*)form_sign->iv, form_sign->iv_len,
                                              (const uint8_t*)form_sign->tag, form_sign->tag_len);
                }
            }
        }
    }

    if (err == SDC_OK) {
        *out_data = internal_data;
        *out_len = internal_data_len;
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_VERIFY, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_verify_formatted_autoload_key_with_secret_mod(
    sdc_session_t *session,
    const uint8_t *formatted_data, const size_t formatted_len,
    const uint8_t *secret_mod_data, const size_t secret_mod_len)
{
    return sdc_intern_formatted_autoload_key_with_secret_mod(
               session,
               formatted_data, formatted_len,
               secret_mod_data, secret_mod_len);
}

sdc_error_t sdc_verify_formatted_autoload_key(sdc_session_t *session,
                                              const uint8_t *formatted_data, const size_t formatted_len)
{
    return sdc_verify_formatted_autoload_key_with_secret_mod(
               session,
               formatted_data, formatted_len,
               NULL, 0);
}

sdc_error_t sdc_verify_formatted_extract_type(
    const uint8_t *formatted_data, const size_t formatted_len,
    sdc_sign_verify_type_t **type)
{
    sdc_error_t err = SDC_OK;
    sdc_form_header_generic_t form_header;
    sdc_sign_verify_type_t *tmp = NULL;

    if (SDC_OK != sdc_intern_check_data_input_buffer(formatted_data, formatted_len))
        return SDC_FORMATTED_DATA_INVALID;

    if (!type) {
        return SDC_ALG_MODE_INVALID;
    }

    // initialize
    *type = NULL;

    err = sdc_intern_formatted_read_header (
        &form_header,
        formatted_data, formatted_len);

    if (err == SDC_OK) {
        if (form_header.operation != SDC_FORMATTED_TYPE_SIGN_VERIFY)
            err = SDC_FORMATTED_DATA_INVALID;
    }

    if (err == SDC_OK) {
        err = sdc_sign_verify_type_alloc(&tmp);
    }

    if (err == SDC_OK) {
        err = sdc_sign_verify_type_set_alg(tmp, form_header.sign_verify.alg);
    }

    if (err == SDC_OK) {
        err = sdc_sign_verify_type_set_hash(tmp, form_header.sign_verify.hash);
    }

    if (err == SDC_OK) {
        *type = tmp;
    } else {
        if (tmp != NULL)
            sdc_sign_verify_type_free(tmp);
    }

    return err;
}


/* defined in sdc_op_common.h */
const sdc_sign_verify_type_t *sdc_sign_verify_get_default(void)
{
    return sdc_arch_sign_verify_get_default();
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_type_alloc(sdc_sign_verify_type_t **type)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_sign_verify_type_alloc(type);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_type_free(sdc_sign_verify_type_t *type)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_sign_verify_type_free(type);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_type_set_alg(sdc_sign_verify_type_t *type, sdc_sign_verify_alg_t alg)
{

    if (!type)
        return SDC_ALG_MODE_INVALID;

    if ((alg < SDC_SIGNVER_ALG_FIRST) || (alg >= SDC_SIGNVER_ALG_END))
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_sign_verify_type_set_alg(type, alg);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_type_set_hash(sdc_sign_verify_type_t *type, sdc_sign_verify_hash_t hash)
{

    if (!type)
        return SDC_ALG_MODE_INVALID;

    if ((hash < SDC_SIGNVER_HASH_FIRST) || (hash >= SDC_SIGNVER_HASH_END))
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_sign_verify_type_set_hash(type, hash);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_type_set_opt_bmsk(sdc_sign_verify_type_t *type, uint64_t opt_bmsk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if ((opt_bmsk & ~(SDC_SIGNVER_OPTS_ALL)) != 0)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_sign_verify_type_set_opt_bmsk(type, opt_bmsk);

}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_type_get_alg(const sdc_sign_verify_type_t *type, sdc_sign_verify_alg_t *alg)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!alg)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_sign_verify_type_get_alg(type, alg);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_type_get_hash(const sdc_sign_verify_type_t *type, sdc_sign_verify_hash_t *hash)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!hash)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_sign_verify_type_get_hash(type, hash);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_type_get_opt_bmsk(const sdc_sign_verify_type_t *type, uint64_t *opt_bmsk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!opt_bmsk)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_sign_verify_type_get_opt_bmsk(type, opt_bmsk);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_desc_alloc(sdc_sign_verify_desc_t **desc)
{
    if (desc == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    *desc = malloc(sizeof(sdc_sign_verify_desc_t));
    if (*desc == NULL) {
        return SDC_NO_MEM;
    }

    memset(*desc, 0, sizeof(sdc_sign_verify_desc_t));

    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_desc_free(sdc_sign_verify_desc_t *desc)
{
    if (desc == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    free(desc);

    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_desc_fill (
    sdc_session_t *session,
    const sdc_sign_verify_type_t *type,
    sdc_sign_verify_desc_t *desc)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!desc)
        return SDC_INVALID_PARAMETER;

    //TODO : check if session is valid - after unittest is changed

    return sdc_arch_sign_verify_desc_fill(session, type, desc);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_desc_get_tag(sdc_sign_verify_desc_t *desc,
                                         size_t *min_val,
                                         size_t *max_val,
                                         size_t *mod,
                                         size_t *default_val)
{
    if (!desc)
        return SDC_INVALID_PARAMETER;

    return sdc_intern_range_get_min_max_mod_dflt(&(desc->tag),
                                           min_val,
                                           max_val,
                                           mod,
                                           default_val);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_sign_verify_desc_get_iv(sdc_sign_verify_desc_t *desc,
                                        size_t *min_val,
                                        size_t *max_val,
                                        size_t *mod,
                                        size_t *default_val)
{
    if (!desc)
        return SDC_INVALID_PARAMETER;

    return sdc_intern_range_get_min_max_mod_dflt(&(desc->iv),
                                           min_val,
                                           max_val,
                                           mod,
                                           default_val);
}

sdc_error_t sdc_sign_verify_get_key_key_lens_fmt(
    sdc_sign_verify_type_t *type,
    sdc_key_fmt_t *sup_key_fmt_protect,
    sdc_key_fmt_t *sup_key_fmt_unprotect,
    sdc_key_len_bmsk_t *sup_key_lens,
    sdc_key_len_t *dflt_key_len)
{
    sdc_key_desc_t key_desc;
    sdc_error_t err;

    if (!type)
        return SDC_ALG_MODE_INVALID;

    err = sdc_arch_sign_verify_key_desc_fill(type, &key_desc);

    if (err == SDC_OK)
        err = sdc_intern_key_lens_fmt(&key_desc,
                                      sup_key_fmt_protect, sup_key_fmt_unprotect,
                                      sup_key_lens, dflt_key_len);

    return err;
}

const char* sdc_sign_verify_alg_name (sdc_sign_verify_alg_t alg)
{
    size_t idx;
    size_t elems;

    if ((alg < SDC_SIGNVER_ALG_FIRST) || (alg >= SDC_SIGNVER_ALG_END))
        return NULL;

    idx = alg;
    elems = sizeof(sdc_sign_verify_alg_names) / sizeof(const char *);

    if (idx<elems) {
        return sdc_sign_verify_alg_names[idx];
    }

    return NULL;
}

const char* sdc_sign_verify_hash_name (sdc_sign_verify_hash_t hash)
{
    size_t idx;
    size_t elems;

    if ((hash < SDC_SIGNVER_HASH_FIRST) || (hash >= SDC_SIGNVER_HASH_END))
        return NULL;

    idx = hash;
    elems = sizeof(sdc_sign_verify_hash_names) / sizeof(const char *);

    if (idx<elems) {
        return sdc_sign_verify_hash_names[idx];
    }

    return NULL;
}

static sdc_error_t sdc_session_type_desc_validation(sdc_session_t *session,
                                              const sdc_sign_verify_type_t *type,
                                              const sdc_sign_verify_desc_t *desc)
{
    /* verify inputs */
    if (!session)
        return SDC_SESSION_INVALID;
    if(!type)
        return SDC_ALG_MODE_INVALID;
    if(!desc)
        return SDC_INVALID_PARAMETER;

    return SDC_OK;
}

sdc_error_t sdc_sign_init(sdc_session_t *session,
                          const sdc_sign_verify_type_t *type,
                          const sdc_sign_verify_desc_t *desc,
                          uint8_t **iv, size_t *iv_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    bool explicit_iv = false;
    uint8_t *internal_iv = NULL;
    size_t internal_iv_len = 0;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_SIGN, SDC_NO_OP);

    if (err == SDC_OK) {
        if (sdc_intern_check_init_dgst_tag_iv_output_buffer(iv, iv_len, &internal_iv_len, SDC_IV_USE_DEFAULT, &explicit_iv) != SDC_OK) {
            err = SDC_IV_INVALID;
        }
    }

    if (err == SDC_OK) {
        if (!explicit_iv) {
            if (internal_iv_len == SDC_IV_USE_DEFAULT) {
                internal_iv_len = desc->iv.dflt;
            }
            internal_iv = NULL;
            /* there might be formats without IV */
            if (internal_iv_len != 0) {
                err = sdc_random_gen_buffer(session, internal_iv_len, &internal_iv);
            }
        } else {
            internal_iv = *iv;
            internal_iv_len = *iv_len;
        }
    }

    if (err == SDC_OK)
        err = sdc_sign_verify_common_check_iv(desc, internal_iv_len);

    if (err == SDC_OK)
        err = sdc_common_signverify_init(session, type, desc,
                                     internal_iv, internal_iv_len, true);

    if (err == SDC_OK) {
        if (!explicit_iv) {
            *iv = internal_iv;
            *iv_len = internal_iv_len;
        }
    } else {
        /* clean allocated memory */
        if (!explicit_iv)
            free (internal_iv);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_SIGN, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
        * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_SIGN, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_verify_init(sdc_session_t *session,
                            const sdc_sign_verify_type_t *type,
                            const sdc_sign_verify_desc_t *desc,
                            const uint8_t *iv, size_t iv_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_VERIFY, SDC_NO_OP);

   if(err == SDC_OK) {
        if (sdc_intern_check_tag_iv_input_buffer(iv, iv_len, SDC_IV_USE_DEFAULT) != SDC_OK) {
            err = SDC_IV_INVALID;
        }
    }

    if (err == SDC_OK) {
        err = sdc_sign_verify_common_check_iv(desc, iv_len);
    }

    if (err == SDC_OK)
        err = sdc_common_signverify_init(session, type, desc,
                                     iv, iv_len, false);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_VERIFY, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_VERIFY, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }
    return err;
}

sdc_error_t sdc_sign_update(sdc_session_t *session,
                            const sdc_sign_verify_type_t *type,
                            const sdc_sign_verify_desc_t *desc,
                            const uint8_t *in_data, const size_t in_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_SIGN, SDC_OP_INITILIZED);

    if (err == SDC_OK) {
        if (sdc_intern_check_data_input_buffer(in_data, in_len) != SDC_OK)
            err = SDC_IN_DATA_INVALID;
    }

    if(err == SDC_OK)
        err = sdc_common_signverify_update(session, type, desc,
                                           in_data, in_len,
                                           true);
    /* No need to update sequence statues if operation is success since INIT
     * already had done status change which is enough for farther operations(FINALIZE).
     *
     * If current operation is failed, reset the history of "operation" and "function" parameters
     * since doesn't support retry operation
     */
    if(err != SDC_OK) {
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_SIGN, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_verify_update(sdc_session_t *session,
                              const sdc_sign_verify_type_t *type,
                              const sdc_sign_verify_desc_t *desc,
                              const uint8_t *in_data, const size_t in_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_VERIFY, SDC_OP_INITILIZED);

    if(err == SDC_OK) {
        if (sdc_intern_check_data_input_buffer(in_data, in_len) != SDC_OK)
            err = SDC_IN_DATA_INVALID;
    }

    if(err == SDC_OK)
        err = sdc_common_signverify_update(session, type, desc,
                                           in_data, in_len,
                                           false);
    /* No need to update sequence statues if operation is success since INIT
     * already had done status change which is enough for farther operations(FINALIZE).
     *
     * If current operation is failed, reset the history of "operation" and "function" parameters
     * since doesn't support retry operation
     */
    if(err != SDC_OK) {
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_VERIFY, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }
    return err;
}

sdc_error_t sdc_sign_finalize(sdc_session_t *session,
                              const sdc_sign_verify_type_t *type,
                              const sdc_sign_verify_desc_t *desc,
                              uint8_t **tag_data, size_t *tag_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    uint8_t *internal_tag =  NULL;
    size_t internal_tag_len = 0;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_SIGN, SDC_OP_INITILIZED);

    if(err == SDC_OK) {
        /* initialize - in case failing with error we want to return
         * buffer pointer NULL and buffer len 0 for all not explicitly specified buffers
         */
        if (SDC_OK != sdc_intern_check_init_dgst_tag_iv_output_buffer(tag_data, tag_len, &internal_tag_len, SDC_TAG_USE_DEFAULT, NULL)) {
            err = SDC_TAG_DATA_INVALID;
        }
    }

    if(err == SDC_OK) {
        if (internal_tag_len == SDC_TAG_USE_DEFAULT) {
            internal_tag_len = desc->tag.dflt;
        }
    }

    if (err == SDC_OK) {
        err = sdc_sign_verify_common_check_tag(desc, internal_tag_len);
    }

    if (err == SDC_OK) {
        if (internal_tag_len != 0) {
            internal_tag = malloc(internal_tag_len);
            if (!internal_tag)
                err = SDC_NO_MEM;
        }
    }

    if (err == SDC_OK)
        err = sdc_common_signverify_finalize(session, type, desc,
                                             internal_tag, internal_tag_len,
                                             NULL, 0,
                                             true);

    if (err == SDC_OK) {
        *tag_data = internal_tag;
        *tag_len = internal_tag_len;
    } else {
        /* clean allocated memory */
        free(internal_tag);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_SIGN, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_verify_finalize(sdc_session_t *session,
                                const sdc_sign_verify_type_t *type,
                                const sdc_sign_verify_desc_t *desc,
                                const uint8_t *tag_data, size_t tag_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;

    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_VERIFY, SDC_OP_INITILIZED);

    if(err == SDC_OK) {
        if (SDC_OK != sdc_intern_check_tag_iv_input_buffer(tag_data, tag_len, SDC_TAG_USE_DEFAULT))
            err = SDC_TAG_DATA_INVALID;
    }

    if (err == SDC_OK) {
        err = sdc_sign_verify_common_check_tag(desc, tag_len);
    }

    if (err == SDC_OK)
        err = sdc_common_signverify_finalize(session, type, desc,
                                             NULL, 0,
                                             tag_data, tag_len,
                                             false);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_VERIFY, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}
